/*
 * Copyright (c) 2004-2006 Voltaire Inc.  All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: ibportstate.c 8693 2006-07-26 22:11:14Z halr $
 */

#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <time.h>
#include <string.h>
#include <getopt.h>

#define __BUILD_VERSION_TAG__ 1.1
#include <common.h>
#include <umad.h>
#include <mad.h>

#undef DEBUG
#define	DEBUG	if (verbose>1) IBWARN
#define IBERROR(fmt, args...)	iberror(__FUNCTION__, fmt, ## args)

static int dest_type = IB_DEST_LID;
static int verbose;

static char *argv0 = "ibportstate";

static void
iberror(const char *fn, char *msg, ...)
{
	char buf[512], *s;
	va_list va;
	int n;

	va_start(va, msg);
	n = vsprintf(buf, msg, va);
	va_end(va);
	buf[n] = 0;

	if ((s = strrchr(argv0, '/')))
		argv0 = s + 1;

	if (verbose)
		printf("%s: iberror: [pid %d] %s: failed: %s\n", argv0, getpid(), fn, buf);
	else
		printf("%s: iberror: failed: %s\n", argv0, buf);

	exit(-1);
}

/*******************************************/

static char *
get_node_info(ib_portid_t *dest, char *data, char **argv, int argc)
{
	int node_type;

	if (!smp_query(data, dest, IB_ATTR_NODE_INFO, 0, 0))
		return "smp query nodeinfo failed";

	node_type = mad_get_field(data, 0, IB_NODE_TYPE_F);
	if (node_type == IB_NODE_SWITCH)	/* Switch NodeType ? */
		return 0;
	else
		return "Node type not switch";
}

static char *
get_port_info(ib_portid_t *dest, char *data, char **argv, int argc)
{
	char buf[2048];
	int portnum = 0;

	if (argc > 0)
		portnum = strtol(argv[0], 0, 0);

	if (!smp_query(data, dest, IB_ATTR_PORT_INFO, portnum, 0))
		return "smp query portinfo failed";

	mad_dump_portstates(buf, sizeof buf, data, sizeof data);

	printf("# Port info: %s port %d\n%s", portid2str(dest), portnum, buf);
	return 0;
}

static char * 
set_port_info(ib_portid_t *dest, char *data, char **argv, int argc)
{
	char buf[2048];
	int portnum = 0;

	if (argc > 0)
		portnum = strtol(argv[0], 0, 0);

	if (!smp_set(data, dest, IB_ATTR_PORT_INFO, portnum, 0))
		return "smp set failed";

	mad_dump_portstates(buf, sizeof buf, data, sizeof data);

	printf("\nPort states after set:\n");
	printf("# Port info: %s port %d\n%s", portid2str(dest), portnum, buf);
	return 0;
}

void
usage(void)
{
	char *basename;

	if (!(basename = strrchr(argv0, '/')))
		basename = argv0;
	else
		basename++;

	fprintf(stderr, "Usage: %s [-d(ebug) -e(rr_show) -v(erbose) -D(irect) -G(uid) -s smlid -V(ersion) -C ca_name -P ca_port "
			"-t(imeout) timeout_ms] <dest dr_path|lid|guid> <portnum> [<op>]\n",
			basename);
	fprintf(stderr, "\tsupported ops: enable, disable, query\n");
	fprintf(stderr, "\n\texamples:\n");
	fprintf(stderr, "\t\t%s 3 1 disable\t\t\t# by lid\n", basename);
	fprintf(stderr, "\t\t%s -G 0x2C9000100D051 1 enable\t# by guid\n", basename);
	fprintf(stderr, "\t\t%s -D 0 1\t\t\t# by direct route\n", basename);
	exit(-1);
}

int
main(int argc, char **argv)
{
	int mgmt_classes[3] = {IB_SMI_CLASS, IB_SMI_DIRECT_CLASS, IB_SA_CLASS};
	ib_portid_t portid = {0};
	ib_portid_t *sm_id = 0, sm_portid = {0};
	extern int ibdebug;
	int timeout = 0, udebug = 0;
	char *ca = 0;
	int ca_port = 0;
	int port_op = 0;	/* default to query */
	char *err;
	char data[IB_SMP_DATA_SIZE];

	static char const str_opts[] = "C:P:t:s:devDGVhu";
	static const struct option long_opts[] = {
		{ "C", 1, 0, 'C'},
		{ "P", 1, 0, 'P'},
		{ "debug", 0, 0, 'd'},
		{ "err_show", 0, 0, 'e'},
		{ "verbose", 0, 0, 'v'},
		{ "Direct", 0, 0, 'D'},
		{ "Guid", 0, 0, 'G'},
		{ "timeout", 1, 0, 't'},
		{ "s", 1, 0, 's'},
		{ "Version", 0, 0, 'V'},
		{ "help", 0, 0, 'h'},
		{ "usage", 0, 0, 'u'},
		{ }
	};

	argv0 = argv[0];

	while (1) {
		int ch = getopt_long(argc, argv, str_opts, long_opts, NULL);
		if ( ch == -1 )
			break;
		switch(ch) {
		case 'd':
			ibdebug++;
			madrpc_show_errors(1);
			umad_debug(udebug);
			udebug++;
			break;
		case 'e':
			madrpc_show_errors(1);
			break;
		case 'D':
			dest_type = IB_DEST_DRPATH;
			break;
		case 'G':
			dest_type = IB_DEST_GUID;
			break;
		case 'C':
			ca = optarg;
			break;
		case 'P':
			ca_port = strtoul(optarg, 0, 0);
			break;
		case 's':
			if (ib_resolve_portid_str(&sm_portid, optarg, IB_DEST_LID, 0) < 0)
				IBERROR("can't resolve SM destination port %s", optarg);
			sm_id = &sm_portid;
			break;
		case 't':
			timeout = strtoul(optarg, 0, 0);
			madrpc_set_timeout(timeout);
			break;
		case 'v':
			verbose++;
			break;
		case 'V':
			fprintf(stderr, "%s %s\n", argv0, get_build_version() );
			exit(-1);
		default:
			usage();
			break;
		}
	}
	argc -= optind;
	argv += optind;

	if (argc < 2)
		usage();

	madrpc_init(ca, ca_port, mgmt_classes, 3);

	if (ib_resolve_portid_str(&portid, argv[0], dest_type, sm_id) < 0)
		IBERROR("can't resolve destination port %s", argv[0]);

	/* First, make sure it is a switch port if it is a "set" */
	if (argc >= 3) {
		if (!strcmp(argv[2], "enable"))
			port_op = 1;
		else if (!strcmp(argv[2], "disable"))
			port_op = 2;
	}

	if (port_op)
		if ((err = get_node_info(&portid, data, argv+1, argc-1)))
			IBERROR("smpquery nodeinfo: %s", err);

	printf("Initial port states:\n");
	if ((err = get_port_info(&portid, data, argv+1, argc-1)))
		IBERROR("smpquery portinfo: %s", err);

	/* Only if one of the "set" options is chosen */
	if (port_op) {
		if (port_op == 1)		/* Enable port */
			mad_set_field(data, 0, IB_PORT_PHYS_STATE_F, 2);	/* Polling */
		else if (port_op == 2) {	/* Disable port */
			mad_set_field(data, 0, IB_PORT_STATE_F, 1);             /* Down */
			mad_set_field(data, 0, IB_PORT_PHYS_STATE_F, 3);        /* Disabled */
		}

		if ((err = set_port_info(&portid, data, argv+1, argc-1)))
			IBERROR("smpset portinfo: %s", err);
	}

	exit(0);
}
